<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\MaRegulation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class RegulationController extends Controller
{
    public function index()
    {
        $title = "Regulasi";
        return view("pages.admin.regulation", compact("title"));
    }


    // FOR FRONTEND
    public function homeRegulation($seo)
    {
        $title = "Regulasi - Dinas Energi dan Sumber Daya Mineral Provinsi Jawa Tengah";
        $regulation = MaRegulation::with('maRegulationFiles')->where("seo", $seo)->first();
        return view('pages.front.regulation', compact("title", "regulation"));
    }

    public function homeListRegulation()
    {
        $title = "Regulasi - Dinas Energi dan Sumber Daya Mineral Provinsi Jawa Tengah";
        $regulation = MaRegulation::all();
        return view('pages.front.list-regulation', compact("title", "regulation"));
    }
    // HANDLE API
    public function dataTable(Request $request)
    {
        $query = MaRegulation::query();

        if ($request->query("search")) {
            $searchValue = $request->query("search")['value'];
            $query->where(function ($query) use ($searchValue) {
                $query->where('title', 'like', '%' . $searchValue . '%');
            });
        }

        $recordsFiltered = $query->count();

        $data = $query->orderBy('created_at', 'desc')
            ->skip($request->query('start'))
            ->limit($request->query('length'))
            ->get();

        $output = $data->map(function ($item) {
            $action = "<div class='dropdown-primary dropdown open'>
                            <button class='btn btn-sm btn-primary dropdown-toggle waves-effect waves-light' id='dropdown-{$item->id}' data-toggle='dropdown' aria-haspopup='true' aria-expanded='true'>
                                Aksi
                            </button>
                            <div class='dropdown-menu' aria-labelledby='dropdown-{$item->id}' data-dropdown-out='fadeOut'>
                                <a class='dropdown-item' onclick='return getData(\"{$item->id}\");' href='javascript:void(0);' title='Edit'>Edit</a>
                                <a class='dropdown-item' href='/admin/regulation-file/$item->id/detail' title='Regulation File'>Regulation File</a>
                                <a class='dropdown-item' onclick='return removeData(\"{$item->id}\");' href='javascript:void(0)' title='Hapus'>Hapus</a>
                            </div>
                        </div>";
            $is_url = $item->is_url == 1 ? '
                    <div class="text-center">
                        <span class="label-switch">Active</span>
                    </div>
                    <div class="input-row">
                        <div class="toggle_status on">
                            <input type="checkbox" onclick="return updateStatus(\'' . $item->id . '\', \'NonActive\');" />
                            <span class="slider"></span>
                        </div>
                    </div>' :
                '<div class="text-center">
                        <span class="label-switch">NonActive</span>
                    </div>
                    <div class="input-row">
                        <div class="toggle_status off">
                            <input type="checkbox" onclick="return updateStatus(\'' . $item->id . '\', \'Active\');" />
                            <span class="slider"></span>
                        </div>
                    </div>';
            $item['action'] = $action;
            $item['is_url'] = $is_url;
            return $item;
        });

        $total = MaRegulation::count();
        return response()->json([
            'draw' => $request->query('draw'),
            'recordsFiltered' => $recordsFiltered,
            'recordsTotal' => $total,
            'data' => $output,
        ]);
    }

    public function getDetail($id)
    {
        try {
            $gallery = MaRegulation::find($id);

            if (!$gallery) {
                return response()->json([
                    "status" => "error",
                    "message" => "Data tidak ditemukan",
                ], 404);
            }

            return response()->json([
                "status" => "success",
                "data" => $gallery
            ]);
        } catch (\Exception $err) {
            return response()->json([
                "status" => "error",
                "message" => $err->getMessage()
            ], 500);
        }
    }

    public function create(Request $request)
    {
        try {
            $data = $request->all();

            $validator = Validator::make(
                $data,
                [
                    "title" => "required|string",
                    "is_url" => "nullable|in:0,1",
                ],
                [
                    "title.required" => "Judul harus diisi",
                    "is_url.in" => "Status url tidak valid",
                ]
            );
            if ($validator->fails()) {
                return response()->json([
                    "status" => "error",
                    "message" => $validator->errors()->first(),
                ], 400);
            }

            $data["seo"] = Str::slug($data["title"]);
            MaRegulation::create($data);
            return response()->json([
                "status" => "success",
                "message" => "Data berhasil dibuat"
            ]);
        } catch (\Exception $err) {
            return response()->json([
                "status" => "error",
                "message" => $err->getMessage(),
            ], 500);
        }
    }

    public function update(Request $request)
    {
        try {
            $data = $request->all();

            $validator = Validator::make(
                $data,
                [
                    "id" => "required|integer",
                    "title" => "required|string",
                    "is_url" => "nullable|in:0,1",
                ],
                [
                    "id.required" => "Data ID harus diisi",
                    "id.integer" => "Type ID tidak sesuai",
                    "title.required" => "Judul harus diisi",
                    "is_url.in" => "Status url tidak valid",
                ]
            );
            if ($validator->fails()) {
                return response()->json([
                    "status" => "error",
                    "message" => $validator->errors()->first(),
                ], 400);
            }


            $regulation = MaRegulation::find($data['id']);
            if (!$regulation) {
                return response()->json([
                    "status" => "error",
                    "message" => "Data tidak ditemukan"
                ], 404);
            }

            $data["seo"] = Str::slug($data["title"]);
            $regulation->update($data);
            return response()->json([
                "status" => "success",
                "message" => "Data berhasil diperbarui"
            ]);
        } catch (\Exception $err) {
            return response()->json([
                "status" => "error",
                "message" => $err->getMessage(),
            ], 500);
        }
    }
    public function updateStatus(Request $request)
    {
        try {
            $data = $request->all();
            $rules = [
                "id" => "required|integer",
                "is_url" => "required|in:0,1",
            ];

            $messages = [
                "id.required" => "Data ID harus diisi",
                "id.integer" => "Type ID tidak sesuai",
                "is_url.required" => "Status Url harus diisi",
                "is_url.in" => "Status Url tidak sesuai",
            ];

            $validator = Validator::make($data, $rules, $messages);
            if ($validator->fails()) {
                return response()->json([
                    "status" => "error",
                    "message" => $validator->errors()->first(),
                ], 400);
            }

            $item = MaRegulation::find($data['id']);
            if (!$item) {
                return response()->json([
                    "status" => "error",
                    "message" => "Data tidak ditemukan"
                ], 404);
            }
            $item->update($data);
            return response()->json([
                "status" => "success",
                "message" => "Status berhasil diperbarui"
            ]);
        } catch (\Exception $err) {
            return response()->json([
                "status" => "error",
                "message" => $err->getMessage(),
            ], 500);
        }
    }

    public function destroy(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), ["id" => "required|integer"], [
                "id.required" => "Data ID harus diisi",
                "id.integer" => "Type ID tidak valid"
            ]);

            if ($validator->fails()) {
                return response()->json([
                    "status" => "error",
                    "message" => $validator->errors()->first()
                ], 400);
            }

            $id = $request->id;
            $data = MaRegulation::find($id);
            if (!$data) {
                return response()->json([
                    "status" => "error",
                    "message" => "Data tidak ditemukan"
                ], 404);
            }

            $data->delete();
            return response()->json([
                "status" => "success",
                "message" => "Data berhasil dihapus"
            ]);
        } catch (\Exception $err) {
            return response()->json([
                "status" => "error",
                "message" => $err->getMessage()
            ], 500);
        }
    }

    // API FRONTEND
    public function homeDataTable(Request $request)
    {
        $query = MaRegulation::query();

        if ($request->query("search")) {
            $searchValue = $request->query("search")['value'];
            $query->where(function ($query) use ($searchValue) {
                $query->where('title', 'like', '%' . $searchValue . '%');
            });
        }

        $recordsFiltered = $query->count();

        $data = $query->orderBy('created_at', 'desc')
            ->skip($request->query('start'))
            ->limit($request->query('length'))
            ->get();

        $output = $data->map(function ($item, $index) {
            $item['no'] = $index + 1;
            $item['detail'] = '<a class="badge badge-primary" href="' . route('home-regulation', $item->seo) . '">Detail</a>';
            return $item;
        });

        $total = MaRegulation::count();
        return response()->json([
            'draw' => $request->query('draw'),
            'recordsFiltered' => $recordsFiltered,
            'recordsTotal' => $total,
            'data' => $output,
        ]);
    }
}
